-- ScummVM - Scumm Interpreter
-- Copyright (C) 2005 The ScummVM project
--
-- This program is free software; you can redistribute it and/or
-- modify it under the terms of the GNU General Public License
-- as published by the Free Software Foundation; either version 2
-- of the License, or (at your option) any later version.

-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.

-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.



usbd_warning = [[**WARNING**
USDB.IRX not found. ScummVM will work without this file,
but you won't be able to use a USB mouse or keyboard,
so you can only play using the gamepad.
For more information, read the READ_PS2 file.
]]

hdd_warning = [[ PS2DEV9.IRX not found, no Harddisk support ]]

mandatory_files = {
    "SCUMMVM.ELF",
    "IOMANX.IRX",
    "FILEXIO.IRX",
    "CODYVDFS.IRX",
    "SJPCM.IRX",
}

optional_files = {
    "USBD.IRX",
    "USB_MASS.IRX",
    "PS2MOUSE.IRX",
    "RPCKBD.IRX",
}

hdd_files = {
	"PS2DEV9.IRX",
	"PS2ATAD.IRX",
	"PS2HDD.IRX",
	"PS2FS.IRX",
	"POWEROFF.IRX"
}

system_cnf = "BOOT2 = cdrom0:\\SCUMMVM.ELF;1\r\nVER = 1.10\r\nVMODE = NTSC\r\n"

dos_filename = Regex "^[A-Z0-9~_-]{1,8}(\\.[A-Z0-9~_-]{1,3})?$"
allowed = Regex "[A-Z0-9~_-]"

--
-- This will recursively add a directory tree to the iso,
-- without too much care about what's going on.
--
function r_mkisofs(dirname, l_dirtree)
    local entries, file, r_dirtree, name = {}
    
    for file in dir(dirname) do
	if (file.type == FLAG_DIR) then
	    if (not ((file.name == ".") or (file.name == ".."))) then
		name = normalize_name(file.name, entries)
		table.insert(entries, name)
		r_dirtree = iso:createdir(l_dirtree, name, 16)
		r_dirtree:setbasicsxa()
		if (dirtemplate ~= nil) then
		    r_dirtree:fromdir(dirtemplate)
		end
		r_mkisofs(dirname .. "/" .. file.name, r_dirtree)
	    end
	else
	    name = normalize_name(file.name, entries)
	    table.insert(entries, name)
	    iso:createfile(l_dirtree, name, Input(dirname .. "/" .. file.name)):setbasicsxa()
	    if (dirtemplate ~= nil) then
		r_dirtree:fromdir(dirtemplate)
	    end
	end
    end
end

--
-- This will handle the creation of the root directory tree.
-- Filenames are checked against a list of built-in filenames
-- to ensure we have everything mandatory.
--
function mkroot(dirname, l_dirtree)
    local entries, i, file, r_dirtree, system_buffer, name = {}

    print "Processing root directory..."
    print " - Creating SYSTEM.CNF file."
    system_buffer = Buffer()
    system_buffer:write(system_cnf)    
    iso:createfile(l_dirtree, "SYSTEM.CNF", system_buffer):setbasicsxa()
    system_buffer = nil

    -- process the mandatory files
    for i, file in ipairs(mandatory_files) do
	iso:createfile(l_dirtree, file, Input(file)):setbasicsxa()
	print(" - Adding " .. file .. " file.")
	table.insert(entries, file)
    end

    -- process the optionnal usb support
    if exists("USBD.IRX") then
	for i, file in ipairs(optional_files) do
	    iso:createfile(l_dirtree, file, Input(file)):setbasicsxa()
	    print(" - Adding " .. file .. " file.")
	    table.insert(entries, file)
	end
    else
	print(usbd_warning)
    end
    
    -- hdd files, also optional
    if exists("PS2DEV9.IRX") then
    for i, file in ipairs(hdd_files) do
	    iso:createfile(l_dirtree, file, Input(file)):setbasicsxa()
	    print(" - Adding " .. file .. " file.")
	    table.insert(entries, file)
	end
    else
	print(hdd_warning)
    end
    

    -- now recursively add any subdirectory we have here (not the *files* that is)
    for file in dir(dirname) do
	if (file.type == FLAG_DIR) then
	    if (not ((file.name == ".") or (file.name == ".."))) then
		print("Directory " .. file.name)
		name = normalize_name(file.name, entries)
		table.insert(entries, name)
		r_dirtree = iso:createdir(l_dirtree, name, 16)
		r_dirtree:setbasicsxa()
		if (dirtemplate ~= nil) then
		    r_dirtree:fromdir(dirtemplate)
		end
		r_mkisofs(dirname .. "/" .. file.name, r_dirtree)
	    end
	end
    end
end

function scummvm_mkisofs(dirname)
    local pvd, root, falsesect
    
    falsesect = {}
    
    if (cdutil == nil) then
	iso:foreword_array(falsesect)
	pvd = createpvd_array(falsesect)
    else
	iso:foreword(cdutil)
	pvd = createpvd(cdutil)
	dirtemplate = cdutil:findpath "/"
    end

    pvd.sysid = "PLAYSTATION"
    pvd.volid = "SCUMMVM"
    pvd.volsetid = "ScummVM"
    pvd.pubid = "http://www.scummvm.org/"
    pvd.prepid = ""
    pvd.appid = "CD-Tool"
    pvd.copyright = "See the COPYING file for details."
    pvd.abstract = "PLAYSTATION"
    pvd.biblio = ""
    
    root = iso:setbasics(pvd, 16, 1, 240)

    root:setbasicsxa()

    if (dirtemplate ~= nil) then
	root:fromdir(dirtemplate)
    end
    
    mkroot(dirname, root)
    
    print "Adding dummy sectors..."
    
    for i = 1, 1500, 1 do
	iso:createsector(falsesect, MODE2_FORM1)
    end
    
    for i = 1, 150, 1 do
	iso:createsector(falsesect, MODE2)
    end
    
    iso:close()

    print("All done!")
end

function main()
    scummvm_mkisofs "."
end

function normalize_name(name, entries)
    local reduced_name, retry = nil, 0

    name = string.upper(name)

    if not dos_filename:Match(name) or is_in(name, entries) then
	reduced_name = reduce_filename(name)
	name = reduced_name
	while(is_in(name, entries)) do
	    name = string.sub(reduced_name, 1, 6) .. "~" .. retry
	    retry = retry + 1
	end
    end

    return name
end

function is_in(name, entries)
    local i, file

    for i, file in ipairs(entries) do
	if (file == name) then
	    return true
	end
    end

    return false
end

function reduce_filename(name)
    local c, name_copy, length

    name_copy = name
    name = ""
    length = 0
    for c in string.gfind(name_copy, ".") do
	if (allowed:Match(c)) then
	    name = name .. c
	    length = length + 1
	elseif (c == " ") then
	    name = name .. "_"
	    length = length + 1
	end
	if length == 8 then
	    break
	end
    end

    return name
end
